/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.worldwind.retrieve;

import gov.nasa.worldwind.avlist.AVList;
import gov.nasa.worldwind.formats.dds.DDSCompressor;
import gov.nasa.worldwind.retrieve.HTTPRetriever;
import gov.nasa.worldwind.retrieve.JarRetriever;
import gov.nasa.worldwind.retrieve.LocalRasterServerRetriever;
import gov.nasa.worldwind.retrieve.RetrievalPostProcessor;
import gov.nasa.worldwind.retrieve.Retriever;
import gov.nasa.worldwind.util.ImageUtil;
import gov.nasa.worldwind.util.Logging;
import gov.nasa.worldwind.util.WWIO;
import gov.nasa.worldwind.util.WWUtil;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedByInterruptException;
import java.util.logging.Level;
import javax.imageio.ImageIO;

public abstract class AbstractRetrievalPostProcessor
implements RetrievalPostProcessor {
    protected AVList avList;
    protected Retriever retriever;

    protected abstract File doGetOutputFile();

    public AbstractRetrievalPostProcessor() {
    }

    public AbstractRetrievalPostProcessor(AVList aVList) {
        this.avList = aVList;
    }

    @Override
    public ByteBuffer run(Retriever retriever) {
        if (retriever == null) {
            String string = Logging.getMessage("nullValue.RetrieverIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        this.retriever = retriever;
        if (!retriever.getState().equals("gov.nasa.worldwind.RetrieverStatusSuccessful")) {
            this.handleUnsuccessfulRetrieval();
            return null;
        }
        if (!this.validateResponseCode()) {
            this.handleInvalidResponseCode();
            return null;
        }
        return this.handleSuccessfulRetrieval();
    }

    public Retriever getRetriever() {
        return this.retriever;
    }

    protected void handleUnsuccessfulRetrieval() {
        if (this.getRetriever().getState().equals("gov.nasa.worldwind.RetrieverStatusError")) {
            this.markResourceAbsent();
        }
    }

    protected ByteBuffer handleSuccessfulRetrieval() {
        try {
            return this.handleContent();
        }
        catch (Exception exception) {
            this.handleContentException(exception);
            return null;
        }
    }

    protected boolean validateResponseCode() {
        if (this.getRetriever() instanceof HTTPRetriever) {
            return this.validateHTTPResponseCode();
        }
        if (this.getRetriever() instanceof JarRetriever) {
            return this.validateJarResponseCode();
        }
        return this.getRetriever() instanceof LocalRasterServerRetriever;
    }

    protected boolean validateHTTPResponseCode() {
        HTTPRetriever hTTPRetriever = (HTTPRetriever)this.getRetriever();
        return hTTPRetriever.getResponseCode() == 200;
    }

    protected boolean validateJarResponseCode() {
        JarRetriever jarRetriever = (JarRetriever)this.getRetriever();
        return jarRetriever.getResponseCode() == 200;
    }

    protected void handleInvalidResponseCode() {
        this.markResourceAbsent();
        if (this.isWMSException()) {
            this.handleWMSExceptionContent();
        } else if (this.isPrimaryContentType("text", this.getRetriever().getContentType())) {
            this.logTextBuffer(this.getRetriever().getBuffer());
        }
    }

    protected void markResourceAbsent() {
    }

    protected boolean saveBuffer() throws IOException {
        return this.saveBuffer(null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected boolean saveBuffer(ByteBuffer byteBuffer) throws IOException {
        File file = this.getOutputFile();
        if (file == null) {
            return false;
        }
        if (file.exists() && !this.overwriteExistingFile()) {
            return false;
        }
        Object object = this.getFileLock();
        synchronized (object) {
            WWIO.saveBuffer(byteBuffer != null ? byteBuffer : this.getRetriever().getBuffer(), file);
        }
        return true;
    }

    protected File getOutputFile() {
        File file = this.doGetOutputFile();
        if (file != null && this.isDeleteOnExit(file)) {
            file.deleteOnExit();
        }
        return file;
    }

    protected boolean overwriteExistingFile() {
        return false;
    }

    protected boolean isDeleteOnExit(File file) {
        return !file.exists() && this.avList != null && this.avList.getValue("gov.nasa.worldwind.avkey.DeleteCacheOnExit") != null;
    }

    protected Object getFileLock() {
        return this;
    }

    protected boolean isPrimaryContentType(String string, String string2) {
        if (WWUtil.isEmpty(string2) || WWUtil.isEmpty(string)) {
            return false;
        }
        return string2.trim().toLowerCase().startsWith(string);
    }

    protected boolean isWMSException() {
        String string = this.getRetriever().getContentType();
        if (WWUtil.isEmpty(string)) {
            return false;
        }
        return string.trim().equalsIgnoreCase("application/vnd.ogc.se_xml");
    }

    protected ByteBuffer handleContent() throws IOException {
        String string = this.getRetriever().getContentType();
        if (WWUtil.isEmpty(string)) {
            String string2 = WWIO.getSuffix(this.getRetriever().getName().split(";")[0]);
            if (!WWUtil.isEmpty(string2)) {
                string = WWIO.makeMimeTypeForSuffix(string2);
            }
            if (WWUtil.isEmpty(string)) {
                Logging.logger().severe(Logging.getMessage("nullValue.ContentTypeIsNullOrEmpty"));
                return null;
            }
        }
        string = string.trim().toLowerCase();
        if (this.isWMSException()) {
            return this.handleWMSExceptionContent();
        }
        if (string.contains("zip")) {
            return this.handleZipContent();
        }
        if (this.isPrimaryContentType("text", string)) {
            return this.handleTextContent();
        }
        if (this.isPrimaryContentType("image", string)) {
            return this.handleImageContent();
        }
        if (this.isPrimaryContentType("application", string)) {
            return this.handleApplicationContent();
        }
        return this.handleUnknownContentType();
    }

    protected void handleContentException(Exception exception) {
        if (exception instanceof ClosedByInterruptException) {
            Logging.logger().log(Level.FINE, Logging.getMessage("generic.OperationCancelled", "retrieval post-processing for " + this.getRetriever().getName()), exception);
        } else if (exception instanceof IOException) {
            this.markResourceAbsent();
            Logging.logger().log(Level.SEVERE, Logging.getMessage("generic.ExceptionWhileSavingRetreivedData", this.getRetriever().getName()), exception);
        }
    }

    protected ByteBuffer handleUnknownContentType() {
        Logging.logger().log(Level.WARNING, Logging.getMessage("generic.UnknownContentType", this.getRetriever().getContentType()));
        return null;
    }

    protected ByteBuffer handleTextContent() throws IOException {
        String string = this.getRetriever().getContentType().trim().toLowerCase();
        if (string.contains("xml")) {
            return this.handleXMLContent();
        }
        if (string.contains("html")) {
            return this.handleHTMLContent();
        }
        this.logTextBuffer(this.getRetriever().getBuffer());
        return null;
    }

    protected ByteBuffer handleXMLContent() throws IOException {
        this.logTextBuffer(this.getRetriever().getBuffer());
        return null;
    }

    protected ByteBuffer handleHTMLContent() throws IOException {
        this.logTextBuffer(this.getRetriever().getBuffer());
        return null;
    }

    protected void logTextBuffer(ByteBuffer byteBuffer) {
        if (byteBuffer == null || !byteBuffer.hasRemaining()) {
            return;
        }
        Logging.logger().warning(WWIO.byteBufferToString(byteBuffer, 2048, null));
    }

    protected ByteBuffer handleZipContent() throws IOException {
        File file = this.getOutputFile();
        if (file == null) {
            return null;
        }
        this.saveBuffer();
        return this.getRetriever().getBuffer();
    }

    protected ByteBuffer handleApplicationContent() throws IOException {
        this.saveBuffer();
        return this.getRetriever().getBuffer();
    }

    protected ByteBuffer handleWMSExceptionContent() {
        StringBuilder stringBuilder = new StringBuilder(this.getRetriever().getName());
        stringBuilder.append("\n");
        stringBuilder.append(WWIO.byteBufferToString(this.getRetriever().getBuffer(), 2048, null));
        Logging.logger().warning(stringBuilder.toString());
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected ByteBuffer handleImageContent() throws IOException {
        File file = this.getOutputFile();
        if (file == null || file.exists() && !this.overwriteExistingFile()) {
            return this.getRetriever().getBuffer();
        }
        if (file.getPath().endsWith("dds")) {
            return this.saveDDS();
        }
        BufferedImage bufferedImage = this.transformPixels();
        if (bufferedImage != null) {
            Object object = this.getFileLock();
            synchronized (object) {
                ImageIO.write((RenderedImage)bufferedImage, this.getRetriever().getContentType().split("/")[1], file);
            }
        } else {
            this.saveBuffer();
        }
        return this.getRetriever().getBuffer();
    }

    protected BufferedImage transformPixels() {
        int[] nArray;
        if (this.avList != null && (nArray = (int[])this.avList.getValue("gov.nasa.worldwind.avkey.TransparencyColors")) != null) {
            return ImageUtil.mapTransparencyColors(this.getRetriever().getBuffer(), nArray);
        }
        return null;
    }

    protected ByteBuffer saveDDS() throws IOException {
        ByteBuffer byteBuffer = this.getRetriever().getBuffer();
        if (!this.getRetriever().getContentType().contains("dds")) {
            byteBuffer = this.convertToDDS();
        }
        this.saveBuffer(byteBuffer);
        return byteBuffer;
    }

    protected ByteBuffer convertToDDS() throws IOException {
        BufferedImage bufferedImage = this.transformPixels();
        ByteBuffer byteBuffer = bufferedImage != null ? DDSCompressor.compressImage(bufferedImage) : DDSCompressor.compressImageBuffer(this.getRetriever().getBuffer());
        return byteBuffer;
    }
}

