/*
 * ArrayToScalar.java
 * Created on 04.04.2013, 17:23:31
 *
 * This file is part of JAMS
 * Copyright (C) FSU Jena
 *
 * JAMS is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * JAMS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with JAMS. If not, see <http://www.gnu.org/licenses/>.
 *
 */
package jams.components.data;

import jams.data.*;
import jams.model.*;

/**
 *
 * @author Sven Kralisch <sven.kralisch at uni-jena.de>
 */
@JAMSComponentDescription(
        title = "ArrayToScalar",
        author = "Sven Kralisch",
        description = "Extracts single scalar doubles from an array of double values",
        date = "2020-09-03",
        version = "1.0_3"
)
@VersionComments(entries = {
    @VersionComments.Entry(version = "1.0_0", comment = "Initial version"),
    @VersionComments.Entry(version = "1.0_1", comment = "Extended to handle multiple values"),
    @VersionComments.Entry(version = "1.0_2", comment = "Extended to allow iteration over indices/dataArrays"),
    @VersionComments.Entry(version = "1.0_3", comment = "Extended to allow iteration be enabled/disabled")
})
public class ArrayToScalar extends JAMSComponent {

    /*
     *  Component attributes
     */
    @JAMSVarDescription(
            access = JAMSVarDescription.AccessType.READ,
            description = "array of values to extract from"
    )
    public Attribute.DoubleArray dataArray;

    @JAMSVarDescription(
            access = JAMSVarDescription.AccessType.READWRITE,
            description = "the index of the value to be extracted"
    )
    public Attribute.Integer[] arrayIndex;

    @JAMSVarDescription(
            access = JAMSVarDescription.AccessType.READ,
            description = "optional correction factor",
            defaultValue = "1"
    )
    public Attribute.Double factor;

    @JAMSVarDescription(
            access = JAMSVarDescription.AccessType.READ,
            description = "iterate through array?",
            defaultValue = "false"
    )
    public Attribute.Boolean iterate;

    @JAMSVarDescription(
            access = JAMSVarDescription.AccessType.WRITE,
            description = "extracted value"
    )
    public Attribute.Double[] dataValue;

    @Override
    public void run() {
        for (int i = 0; i < arrayIndex.length; i++) {
            if (arrayIndex[i].getValue() != -1 && arrayIndex[i].getValue() < dataArray.getValue().length) {
                dataValue[i].setValue(dataArray.getValue()[arrayIndex[i].getValue()] * factor.getValue());
            }
        }
        if (iterate.getValue()) {
            for (int i = 0; i < arrayIndex.length; i++) {
                arrayIndex[i].setValue(arrayIndex[i].getValue() + 1);
            }
        }
    }

}
