/*
 * Decompiled with CFR 0.152.
 */
package org.jgrasstools.gears.io.las.index;

import com.vividsolutions.jts.geom.Coordinate;
import com.vividsolutions.jts.geom.CoordinateList;
import com.vividsolutions.jts.geom.Envelope;
import com.vividsolutions.jts.geom.Geometry;
import com.vividsolutions.jts.geom.GeometryFactory;
import com.vividsolutions.jts.geom.LinearRing;
import com.vividsolutions.jts.geom.MultiPoint;
import com.vividsolutions.jts.geom.Polygon;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import oms3.annotations.Author;
import oms3.annotations.Description;
import oms3.annotations.Execute;
import oms3.annotations.Finalize;
import oms3.annotations.In;
import oms3.annotations.Keywords;
import oms3.annotations.Label;
import oms3.annotations.License;
import oms3.annotations.Name;
import oms3.annotations.Status;
import oms3.annotations.UI;
import org.geotools.coverage.grid.GridCoordinates2D;
import org.geotools.coverage.grid.GridGeometry2D;
import org.geotools.data.simple.SimpleFeatureCollection;
import org.geotools.feature.DefaultFeatureCollection;
import org.geotools.feature.simple.SimpleFeatureBuilder;
import org.geotools.feature.simple.SimpleFeatureTypeBuilder;
import org.geotools.geometry.DirectPosition2D;
import org.geotools.geometry.Envelope2D;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.geotools.geometry.jts.ReferencedEnvelope3D;
import org.geotools.referencing.CRS;
import org.jgrasstools.gears.io.las.core.ALasReader;
import org.jgrasstools.gears.io.las.core.ALasWriter;
import org.jgrasstools.gears.io.las.core.ILasHeader;
import org.jgrasstools.gears.io.las.core.LasRecord;
import org.jgrasstools.gears.io.las.index.strtree.STRtreeJGT;
import org.jgrasstools.gears.libs.exceptions.ModelsIllegalargumentException;
import org.jgrasstools.gears.libs.modules.JGTModel;
import org.jgrasstools.gears.modules.utils.fileiterator.OmsFileIterator;
import org.jgrasstools.gears.utils.CrsUtilities;
import org.jgrasstools.gears.utils.coverage.CoverageUtilities;
import org.jgrasstools.gears.utils.files.FileUtilities;
import org.jgrasstools.gears.utils.geometry.GeometryUtilities;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.geometry.DirectPosition;
import org.opengis.referencing.crs.CoordinateReferenceSystem;

@Description(value="Creates indexes for Las files.")
@Author(name="Andrea Antonello", contact="www.hydrologis.com")
@Keywords(value="las, lidar")
@Label(value="Lesto/utilities")
@Name(value="lasindexer")
@Status(value=5)
@License(value="http://www.gnu.org/licenses/gpl-3.0.html")
public class LasIndexer
extends JGTModel {
    public static final String INDEX_LASFOLDER = "index.lasfolder";
    @Description(value="The folder containing the las files to index.")
    @UI(value="infolder")
    @In
    public String inFolder;
    @Description(value="The name for the main index file.")
    @In
    public String pIndexname = "index.lasfolder";
    @Description(value="The optional code defining the target coordinate reference system. This is needed only if the file has no prj file. If set, it will be used over the prj file.")
    @UI(value="crs")
    @In
    public String pCode;
    @Description(value="The size of the cells into which to split the las file for indexing (in units defined by the projection).")
    @In
    public double pCellsize = 5.0;
    @Description(value="Create overview shapefile (this creates a convexhull of the points).")
    @In
    public boolean doOverview = false;
    @Description(value="The number of threads to use for the process.")
    @In
    public int pThreads = 1;
    private CoordinateReferenceSystem crs;
    private ConcurrentLinkedQueue<Polygon> envelopesQueue;

    @Execute
    public void process() throws Exception {
        this.checkNull(this.inFolder, this.pIndexname);
        if (this.pCellsize <= 0.0) {
            throw new ModelsIllegalargumentException("The cell size parameter needs to be > 0.", this);
        }
        if (!new File(this.inFolder).exists()) {
            throw new ModelsIllegalargumentException("The inFolder parameter has to be valid.", this);
        }
        try {
            if (this.pCode != null) {
                this.crs = CRS.decode((String)this.pCode);
            }
        }
        catch (Exception e1) {
            throw new ModelsIllegalargumentException("An error occurred while reading the projection definition: " + e1.getLocalizedMessage(), this);
        }
        this.pm.message("Las files to be added to the index:");
        OmsFileIterator iter = new OmsFileIterator();
        iter.inFolder = this.inFolder;
        iter.fileFilter = new FileFilter(){

            @Override
            public boolean accept(File file) {
                String name = file.getName();
                if (name.endsWith("indexed.las")) {
                    return false;
                }
                boolean isLas = name.toLowerCase().endsWith(".las");
                if (isLas) {
                    LasIndexer.this.pm.message("   " + name);
                }
                return isLas;
            }
        };
        iter.process();
        List<File> filesList = iter.filesList;
        this.pm.beginTask("Creating readers index...", filesList.size());
        STRtreeJGT mainTree = new STRtreeJGT();
        for (File file : filesList) {
            ALasReader reader = ALasReader.getReader(file, this.crs);
            Object object = null;
            try {
                reader.open();
                ILasHeader header = reader.getHeader();
                if (this.crs == null) {
                    this.crs = header.getCrs();
                }
                ReferencedEnvelope3D envelope = header.getDataEnvelope();
                File newLasFile = this.getNewLasFile(file);
                mainTree.insert((Envelope)envelope, newLasFile.getName());
            }
            catch (Throwable header) {
                object = header;
                throw header;
            }
            finally {
                if (reader != null) {
                    if (object != null) {
                        try {
                            reader.close();
                        }
                        catch (Throwable header) {
                            ((Throwable)object).addSuppressed(header);
                        }
                    } else {
                        reader.close();
                    }
                }
            }
            this.pm.worked(1);
        }
        this.pm.done();
        File mainIndex = new File(this.inFolder, this.pIndexname);
        byte[] mainIndexBytes = LasIndexer.serialize(mainTree);
        LasIndexer.dumpBytes(mainIndex, mainIndexBytes);
        CrsUtilities.writeProjectionFile(mainIndex.getAbsolutePath(), "lasfolder", this.crs);
        if (this.doOverview) {
            this.envelopesQueue = new ConcurrentLinkedQueue();
        }
        if (this.pThreads > 1) {
            ExecutorService fixedThreadPool = Executors.newFixedThreadPool(this.pThreads);
            for (final File file : filesList) {
                Runnable runner = new Runnable(){

                    @Override
                    public void run() {
                        try {
                            LasIndexer.this.processFile(file, true);
                        }
                        catch (Exception e) {
                            LasIndexer.this.pm.errorMessage("Problems indexing file: " + file.getName());
                            e.printStackTrace();
                        }
                    }
                };
                fixedThreadPool.execute(runner);
            }
            try {
                fixedThreadPool.shutdown();
                fixedThreadPool.awaitTermination(30L, TimeUnit.DAYS);
                fixedThreadPool.shutdownNow();
            }
            catch (InterruptedException e) {
                e.printStackTrace();
            }
        } else {
            for (File file : filesList) {
                this.processFile(file, false);
            }
        }
        if (this.doOverview) {
            File overviewFile = FileUtilities.substituteExtention(mainIndex, "shp");
            SimpleFeatureTypeBuilder b = new SimpleFeatureTypeBuilder();
            b.setName("overview");
            b.setCRS(this.crs);
            b.add("the_geom", Polygon.class);
            b.add("file", String.class);
            SimpleFeatureType type = b.buildFeatureType();
            SimpleFeatureBuilder builder = new SimpleFeatureBuilder(type);
            DefaultFeatureCollection overviewFC = new DefaultFeatureCollection();
            for (Polygon polygon : this.envelopesQueue) {
                Object[] values = new Object[]{polygon, polygon.getUserData()};
                builder.addAll(values);
                SimpleFeature feature = builder.buildFeature(null);
                overviewFC.add(feature);
            }
            this.dumpVector((SimpleFeatureCollection)overviewFC, overviewFile.getAbsolutePath());
        }
    }

    private void processFile(File file, boolean isMultiThreaded) throws Exception {
        String name = file.getName();
        File newLasFile = this.getNewLasFile(file);
        File indexFile = this.getNetIndexFile(file);
        if (indexFile.exists() && newLasFile.exists()) {
            this.pm.message("Index existing already for file: " + name);
            return;
        }
        if (indexFile.exists() || newLasFile.exists()) {
            indexFile.delete();
            newLasFile.delete();
        }
        this.pm.message("Processing file: " + name);
        CoordinateList pointsList = new CoordinateList();
        try (ALasReader reader = ALasReader.getReader(file, this.crs);){
            reader.open();
            ILasHeader header = reader.getHeader();
            long recordsCount = header.getRecordsCount();
            if (recordsCount == 0L) {
                this.pm.errorMessage("No points found in: " + name);
                return;
            }
            ReferencedEnvelope3D envelope = header.getDataEnvelope();
            ReferencedEnvelope env2d = new ReferencedEnvelope((ReferencedEnvelope)envelope);
            Envelope2D e = new Envelope2D((org.opengis.geometry.Envelope)env2d);
            double north = e.getMaxY();
            double south = e.getMinY();
            double east = e.getMaxX();
            double west = e.getMinX();
            int cols = (int)Math.round(e.getWidth() / this.pCellsize);
            int rows = (int)Math.round(e.getHeight() / this.pCellsize);
            double xRes = e.getWidth() / (double)cols;
            double yRes = e.getHeight() / (double)rows;
            double width = (east += xRes / 2.0) - (west -= xRes / 2.0);
            double height = (north += yRes / 2.0) - (south -= yRes / 2.0);
            cols = (int)Math.round(width / this.pCellsize);
            rows = (int)Math.round(height / this.pCellsize);
            xRes = width / (double)cols;
            yRes = height / (double)rows;
            this.pm.message("Splitting " + name + " into tiles of " + (float)xRes + " x " + (float)yRes + ".");
            GridGeometry2D gridGeometry = CoverageUtilities.gridGeometryFromRegionValues(north, south, east, west, cols, rows, reader.getHeader().getCrs());
            ArrayList[][] dotOnMatrix = new ArrayList[cols][rows];
            if (!isMultiThreaded) {
                this.pm.beginTask("Sorting points for " + name, (int)recordsCount);
            } else {
                this.pm.message("Sorting points for " + name + "...");
            }
            while (reader.hasNextPoint()) {
                LasRecord dot = reader.getNextPoint();
                DirectPosition2D wPoint = new DirectPosition2D(dot.x, dot.y);
                GridCoordinates2D gridCoord = gridGeometry.worldToGrid((DirectPosition)wPoint);
                int x = gridCoord.x;
                int y = gridCoord.y;
                if (dotOnMatrix[x][y] == null) {
                    dotOnMatrix[x][y] = new ArrayList();
                }
                dotOnMatrix[x][y].add(dot);
                if (this.doOverview) {
                    pointsList.add((Object)new Coordinate(dot.x, dot.y));
                }
                if (isMultiThreaded) continue;
                this.pm.worked(1);
            }
            if (!isMultiThreaded) {
                this.pm.done();
            }
            try (ALasWriter writer = ALasWriter.getWriter(newLasFile, reader.getHeader().getCrs());){
                writer.setBounds(reader.getHeader());
                writer.open();
                int addedTiles = 0;
                STRtreeJGT tree = new STRtreeJGT();
                if (!isMultiThreaded) {
                    this.pm.beginTask("Write and index new las...", cols);
                } else {
                    this.pm.message("Write and index new las...");
                }
                long pointCount = 0L;
                for (int c = 0; c < cols; ++c) {
                    for (int r = 0; r < rows; ++r) {
                        ArrayList dotsList = dotOnMatrix[c][r];
                        if (dotsList == null || dotsList.size() == 0) continue;
                        Coordinate coord = CoverageUtilities.coordinateFromColRow(c, r, gridGeometry);
                        Envelope env = new Envelope(coord);
                        env.expandBy(xRes / 2.0, yRes / 2.0);
                        long tmpCount = pointCount;
                        double avgElevValue = 0.0;
                        double avgIntensityValue = 0.0;
                        int count = 0;
                        for (LasRecord dot : dotsList) {
                            writer.addPoint(dot);
                            ++pointCount;
                            avgElevValue += dot.z;
                            avgIntensityValue += (double)dot.intensity;
                            ++count;
                        }
                        tree.insert(env, new double[]{tmpCount, pointCount, avgElevValue /= (double)count, avgIntensityValue /= (double)count});
                        ++addedTiles;
                    }
                    if (isMultiThreaded) continue;
                    this.pm.worked(1);
                }
                if (!isMultiThreaded) {
                    this.pm.done();
                }
                byte[] serialized = LasIndexer.serialize(tree);
                LasIndexer.dumpBytes(indexFile, serialized);
                this.pm.message("Tiles added for " + name + ": " + addedTiles);
            }
        }
        if (this.doOverview) {
            this.pm.message("Create overview for " + name);
            MultiPoint multiPoint = this.gf.createMultiPoint(pointsList.toCoordinateArray());
            Geometry polygon = multiPoint.convexHull();
            polygon.setUserData((Object)name);
            this.envelopesQueue.add((Polygon)polygon);
        }
    }

    private File getNetIndexFile(File file) {
        String nameWithoutExtention = FileUtilities.getNameWithoutExtention(file);
        File indexFile = new File(file.getParentFile(), nameWithoutExtention + "_indexed.lasfix");
        return indexFile;
    }

    private File getNewLasFile(File file) {
        String nameWithoutExtention = FileUtilities.getNameWithoutExtention(file);
        File newLasFile = new File(file.getParentFile(), nameWithoutExtention + "_indexed.las");
        return newLasFile;
    }

    public static Polygon envelopeToPolygon(Envelope envelope) {
        double w = envelope.getMinX();
        double e = envelope.getMaxX();
        double s = envelope.getMinY();
        double n = envelope.getMaxY();
        Coordinate[] coords = new Coordinate[]{new Coordinate(w, n), new Coordinate(e, n), new Coordinate(e, s), new Coordinate(w, s), new Coordinate(w, n)};
        GeometryFactory gf = GeometryUtilities.gf();
        LinearRing linearRing = gf.createLinearRing(coords);
        Polygon polygon = gf.createPolygon(linearRing, null);
        return polygon;
    }

    @Finalize
    public void close() throws Exception {
    }

    private static byte[] serialize(Object obj) throws IOException {
        try (ByteArrayOutputStream bos = new ByteArrayOutputStream();){
            ObjectOutputStream out = new ObjectOutputStream(bos);
            out.writeObject(obj);
            out.close();
            byte[] byArray = bos.toByteArray();
            return byArray;
        }
    }

    private static void dumpBytes(File file, byte[] bytes) throws Exception {
        try (RandomAccessFile raf = new RandomAccessFile(file, "rw");){
            raf.write(bytes);
        }
    }
}

