/*
 * Decompiled with CFR 0.152.
 */
package jams.components.io;

import jams.components.efficiencies.Regression;
import jams.data.Attribute;
import jams.model.JAMSComponent;
import jams.model.JAMSComponentDescription;
import jams.model.JAMSVarDescription;
import jams.model.VersionComments;
import jams.workspace.DataSetDefinition;
import jams.workspace.DataValue;
import jams.workspace.DefaultDataSet;
import jams.workspace.stores.InputDataStore;
import jams.workspace.stores.TSDataStore;
import java.time.LocalDate;
import java.time.Year;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.GregorianCalendar;

@JAMSComponentDescription(title="TSDataStoreReader", author="Sven Kralisch", date="2025-02-01", version="1.3", description="This component can be used to obtain data from a time series data store which contains only double values and a number of station-specific metadata. Additional functions:\n- automated time shift if start date of datastore is before start date of model\n- automated aggregation if time steps of data store and model differ")
@VersionComments(entries={@VersionComments.Entry(version="1.0_0", date="2008-10-20", comment="Initial version"), @VersionComments.Entry(version="1.0_1", date="2013-06-17", comment="Cache functions removed, minor bug fixes"), @VersionComments.Entry(version="1.1", date="2014-02-16", comment="- Aggregation functions if time steps of data store and model differ\n- Fixed wrong time shift in case of monthly data\n"), @VersionComments.Entry(version="1.1_1", date="2014-05-14", comment="Fixed bug that caused wrong forward skipping if time offset was very long (> 68 years of daily data)"), @VersionComments.Entry(version="1.2", date="2014-06-20", comment="Added attributes to output column names and columns IDs for further use"), @VersionComments.Entry(version="1.3", date="2025-02-01", comment="Added option to switch between three calendar types (Gregorian, Gregorian w/o leap days, 360-days). Works only for daily time steps.")})
public class TSDataStoreReader
extends JAMSComponent {
    @JAMSVarDescription(access=JAMSVarDescription.AccessType.READ, description="Datastore ID")
    public Attribute.String id;
    @JAMSVarDescription(access=JAMSVarDescription.AccessType.WRITE, description="Descriptive name of the dataset (equals datastore ID)")
    public Attribute.String dataSetName;
    @JAMSVarDescription(access=JAMSVarDescription.AccessType.WRITE, description="Array of double values received from the datastore. Order according to datastore")
    public Attribute.DoubleArray dataArray;
    @JAMSVarDescription(access=JAMSVarDescription.AccessType.WRITE, description="Array of column names")
    public Attribute.StringArray name;
    @JAMSVarDescription(access=JAMSVarDescription.AccessType.WRITE, description="Array of column IDs")
    public Attribute.StringArray columnID;
    @JAMSVarDescription(access=JAMSVarDescription.AccessType.WRITE, description="Array of station elevations")
    public Attribute.DoubleArray elevation;
    @JAMSVarDescription(access=JAMSVarDescription.AccessType.WRITE, description="Array of station's x coordinate")
    public Attribute.DoubleArray xCoord;
    @JAMSVarDescription(access=JAMSVarDescription.AccessType.WRITE, description="Array of station's y coordinate")
    public Attribute.DoubleArray yCoord;
    @JAMSVarDescription(access=JAMSVarDescription.AccessType.WRITE, description="Regression coefficients")
    public Attribute.DoubleArray regCoeff;
    @JAMSVarDescription(access=JAMSVarDescription.AccessType.READ, description="The time interval within which the component shall read data from the datastore")
    public Attribute.TimeInterval timeInterval;
    @JAMSVarDescription(access=JAMSVarDescription.AccessType.READ, description="Aggregate multiple datastore entries to averages or sums?", defaultValue="true")
    public Attribute.Boolean calcAvg;
    @JAMSVarDescription(access=JAMSVarDescription.AccessType.READ, description="The current model time - needed in case of aggregation over irregular time steps (e.g. months). Aggregation is disabled if this value is not set.")
    public Attribute.Calendar time;
    @JAMSVarDescription(access=JAMSVarDescription.AccessType.READ, description="Lines skipped to reach timeInterval start")
    public Attribute.Integer skipLines;
    @JAMSVarDescription(access=JAMSVarDescription.AccessType.READ, description="Type of calendar:\n0 - standard calendar\n1 - 360-day calendar\n2 - standard calendar w/o leap years", defaultValue="0")
    public Attribute.Integer calendarType;
    private TSDataStore store;
    private double[] doubles;
    private double[] elevationArray;
    boolean shifted = false;
    int tsRatio = 1;
    Attribute.Calendar storeDate;
    int storeUnit;
    int storeUnitCount;
    int targetUnit;
    int targetUnitCount;

    public void init() {
        this.shifted = false;
        InputDataStore is = null;
        if (this.id != null) {
            is = this.getModel().getWorkspace().getInputDataStore(this.id.getValue());
        }
        if (is == null) {
            this.getModel().getRuntime().sendHalt("Error accessing datastore \"" + this.id + "\" from " + this.getInstanceName() + ": Datastore could not be found!");
            return;
        }
        if (!(is instanceof TSDataStore)) {
            this.getModel().getRuntime().sendHalt("Error accessing datastore \"" + this.id + "\" from " + this.getInstanceName() + ": Datastore is not a time series datastore!");
            return;
        }
        this.store = (TSDataStore)is;
        if (this.store.getStartDate().after(this.timeInterval.getStart()) && this.store.getStartDate().compareTo(this.timeInterval.getStart(), this.timeInterval.getTimeUnit()) != 0) {
            this.getModel().getRuntime().sendHalt("Error accessing datastore \"" + this.id + "\" from " + this.getInstanceName() + ": Start date of datastore (" + this.store.getStartDate() + ") does not match given time interval (" + this.timeInterval.getStart() + ")!");
            return;
        }
        Attribute.Calendar storeEnd = this.store.getEndDate();
        if (this.calendarType.getValue() == 1 && storeEnd.get(5) == 30) {
            storeEnd.add(6, 1);
        }
        if (storeEnd.before(this.timeInterval.getEnd()) && this.store.getEndDate().compareTo(this.timeInterval.getEnd(), this.timeInterval.getTimeUnit()) != 0) {
            this.getModel().getRuntime().sendHalt("Error accessing datastore \"" + this.id + "\" from " + this.getInstanceName() + ": End date of datastore (" + this.store.getEndDate() + ") does not match given time interval (" + this.timeInterval.getEnd() + ")!");
            return;
        }
        DataSetDefinition dsDef = this.store.getDataSetDefinition();
        if (dsDef.getAttributeValues("X") == null) {
            this.getModel().getRuntime().sendHalt("Error in data set definition \"" + this.id + "\" from " + this.getInstanceName() + ": x coordinate not specified");
        }
        if (dsDef.getAttributeValues("Y") == null) {
            this.getModel().getRuntime().sendHalt("Error in data set definition \"" + this.id + "\" from " + this.getInstanceName() + ": y coordinate not specified");
        }
        if (dsDef.getAttributeValues("ELEVATION") == null) {
            this.getModel().getRuntime().sendHalt("Error in data set definition \"" + this.id + "\" from " + this.getInstanceName() + ": elevation not specified");
        }
        this.xCoord.setValue(this.listToDoubleArray(dsDef.getAttributeValues("X")));
        this.yCoord.setValue(this.listToDoubleArray(dsDef.getAttributeValues("Y")));
        this.elevation.setValue(this.listToDoubleArray(dsDef.getAttributeValues("ELEVATION")));
        this.name.setValue(this.listToStringArray(dsDef.getAttributeValues("NAME")));
        this.columnID.setValue(this.listToStringArray(dsDef.getAttributeValues("ID")));
        this.elevationArray = this.elevation.getValue();
        this.dataSetName.setValue(this.id.getValue());
        this.getModel().getRuntime().println("Datastore " + this.id + " initialized!", 3);
        this.doubles = new double[this.store.getDataSetDefinition().getColumnCount()];
        this.dataArray.setValue(this.doubles);
    }

    private double[] listToDoubleArray(ArrayList<Object> list) {
        if (list == null) {
            return null;
        }
        double[] result = new double[list.size()];
        int i = 0;
        for (Object o : list) {
            result[i] = (Double)o;
            ++i;
        }
        return result;
    }

    private String[] listToStringArray(ArrayList<Object> list) {
        if (list == null) {
            return null;
        }
        String[] result = new String[list.size()];
        int i = 0;
        for (Object o : list) {
            result[i] = o.toString();
            ++i;
        }
        return result;
    }

    private void checkConsistency() {
        Attribute.Calendar targetDate = this.timeInterval.getStart().clone();
        this.targetUnit = this.timeInterval.getTimeUnit();
        this.targetUnitCount = this.timeInterval.getTimeUnitCount();
        this.storeDate = this.store.getStartDate().clone();
        this.storeUnit = this.store.getTimeUnit();
        this.storeUnitCount = this.store.getTimeUnitCount();
        int leapOffset = 0;
        if (this.calendarType.getValue() == 2) {
            leapOffset = TSDataStoreReader.countLeapDays(this.storeDate, targetDate);
        }
        if (this.skipLines == null) {
            this.storeDate.removeUnsignificantComponents(this.storeUnit);
            targetDate.removeUnsignificantComponents(this.targetUnit);
            int offset = this.storeDate.compareTo(targetDate, this.targetUnit);
            if (offset > 0) {
                this.getModel().getRuntime().sendHalt("Time series data read by " + this.getInstanceName() + " start after model start time!\n(" + this.store.getStartDate() + " vs " + this.timeInterval.getStart() + ")");
            } else if (offset < 0) {
                int steps;
                if (this.calendarType.getValue() == 0 || this.calendarType.getValue() == 2) {
                    long diff = (targetDate.getTimeInMillis() - this.storeDate.getTimeInMillis()) / 1000L;
                    switch (this.storeUnit) {
                        case 6: {
                            steps = (int)(diff / 3600L / 24L / (long)this.storeUnitCount);
                            this.storeDate.add(this.storeUnit, this.storeUnitCount * steps);
                            break;
                        }
                        case 11: {
                            steps = (int)(diff / 3600L / (long)this.storeUnitCount);
                            this.storeDate.add(this.storeUnit, this.storeUnitCount * steps);
                            break;
                        }
                        case 3: {
                            steps = (int)(diff / 3600L / 24L / 7L / (long)this.storeUnitCount);
                            this.storeDate.add(this.storeUnit, this.storeUnitCount * steps);
                            break;
                        }
                        case 12: {
                            steps = (int)(diff / 60L / (long)this.storeUnitCount);
                            this.storeDate.add(this.storeUnit, this.storeUnitCount * steps);
                            break;
                        }
                        case 13: {
                            steps = (int)(diff / (long)this.storeUnitCount);
                            this.storeDate.add(this.storeUnit, this.storeUnitCount * steps);
                            break;
                        }
                        default: {
                            steps = this.iterateStoreDate(targetDate);
                        }
                    }
                    steps -= leapOffset;
                } else {
                    CustomCalendar360 storeDate360;
                    CustomCalendar360 targetDate360 = new CustomCalendar360(targetDate.get(1), targetDate.get(2) + 1, targetDate.get(5));
                    steps = (int)targetDate360.calculateOffset(storeDate360 = new CustomCalendar360(this.storeDate.get(1), this.storeDate.get(2) + 1, this.storeDate.get(5)));
                    if (steps < 0) {
                        this.getModel().getRuntime().sendHalt("Time series data read by " + this.getInstanceName() + " start after model start time!\n(" + this.store.getStartDate() + " vs " + this.timeInterval.getStart() + ")");
                    }
                }
                this.store.skip(steps);
            }
        } else {
            this.store.skip(this.skipLines.getValue());
        }
        if (this.storeUnit != this.targetUnit || this.storeUnitCount != this.targetUnitCount) {
            if (this.time == null) {
                this.getModel().getRuntime().sendHalt("Time steps in datastore " + this.store.getID() + " and model are different while time is not set! Please set the time atrtibute or adapt your datastore");
            }
            if (this.storeUnit > 2 && this.targetUnit > 2) {
                int storeMS = this.getMilliseconds(this.storeUnit);
                int targetMS = this.getMilliseconds(this.targetUnit);
                double dRatio = (double)(targetMS * this.targetUnitCount) / (double)(storeMS * this.storeUnitCount);
                int ratio = (int)Math.floor(dRatio);
                if ((double)ratio != dRatio) {
                    this.getModel().getRuntime().sendHalt("Time steps in datastore " + this.store.getID() + " and model are incompatible. Please adapt your datastore first!");
                }
                this.tsRatio = ratio;
            } else {
                this.tsRatio = -1;
            }
        }
    }

    private int getMilliseconds(int unit) {
        int ms = 0;
        switch (unit) {
            case 6: {
                ms = 86400000;
                break;
            }
            case 11: {
                ms = 3600000;
                break;
            }
            case 3: {
                ms = 604800000;
                break;
            }
            case 12: {
                ms = 60000;
                break;
            }
            case 13: {
                ms = 1000;
                break;
            }
            case 14: {
                ms = 1;
                break;
            }
            default: {
                this.getModel().getRuntime().sendHalt("Cannot calculate constant time unit duration!");
            }
        }
        return ms;
    }

    private int iterateStoreDate(Attribute.Calendar date) {
        int steps = 0;
        while (this.storeDate.compareTo(date, this.storeUnit) < 0) {
            this.storeDate.add(this.storeUnit, this.storeUnitCount);
            ++steps;
        }
        return steps;
    }

    public void initAll() {
        this.checkConsistency();
    }

    public void run() {
        if (this.tsRatio == 1) {
            if (this.calendarType.getValue() == 0) {
                DefaultDataSet ds = this.store.getNext();
                if (ds == null) {
                    this.getModel().getRuntime().sendHalt("Empty dataset found in component " + this.getInstanceName() + " (" + this.time + ")");
                }
                DataValue[] data = ds.getData();
                for (int i = 1; i < data.length; ++i) {
                    this.doubles[i - 1] = data[i].getDouble();
                }
            } else if (this.calendarType.getValue() == 1) {
                int year = this.time.get(1);
                boolean isLeap = Year.of(year).isLeap();
                int day = this.time.get(6);
                if (isLeap) {
                    --day;
                }
                if (!(isLeap && day == 59 || day == 151 || day == 212 || day == 243 || day == 304 || day == 365)) {
                    DefaultDataSet ds = this.store.getNext();
                    if (ds == null) {
                        this.getModel().getRuntime().sendHalt("Empty dataset found in component " + this.getInstanceName() + " (" + this.time + ")");
                    }
                    DataValue[] data = ds.getData();
                    for (int i = 1; i < data.length; ++i) {
                        this.doubles[i - 1] = data[i].getDouble();
                    }
                }
            } else {
                int year = this.time.get(1);
                boolean isLeap = Year.of(year).isLeap();
                int day = this.time.get(6);
                if (!isLeap || day != 60) {
                    DefaultDataSet ds = this.store.getNext();
                    if (ds == null) {
                        this.getModel().getRuntime().sendHalt("Empty dataset found in component " + this.getInstanceName() + " (" + this.time + ")");
                    }
                    DataValue[] data = ds.getData();
                    for (int i = 1; i < data.length; ++i) {
                        this.doubles[i - 1] = data[i].getDouble();
                    }
                }
            }
            this.dataArray.setValue(this.doubles);
            this.regCoeff.setValue(Regression.calcLinReg(this.elevationArray, this.doubles));
        } else {
            int i;
            int n;
            if (this.tsRatio < 0) {
                Attribute.Calendar nextTime = this.time.clone();
                nextTime.add(this.targetUnit, this.targetUnitCount);
                n = this.iterateStoreDate(nextTime);
            } else {
                n = this.tsRatio;
            }
            for (i = 0; i < this.doubles.length; ++i) {
                this.doubles[i] = 0.0;
            }
            for (int j = 0; j < n; ++j) {
                DefaultDataSet ds = this.store.getNext();
                DataValue[] data = ds.getData();
                for (int i2 = 1; i2 < data.length; ++i2) {
                    int n2 = i2 - 1;
                    this.doubles[n2] = this.doubles[n2] + data[i2].getDouble();
                }
            }
            if (this.calcAvg.getValue()) {
                i = 0;
                while (i < this.doubles.length) {
                    int n3 = i++;
                    this.doubles[n3] = this.doubles[n3] / (double)n;
                }
            }
            this.dataArray.setValue(this.doubles);
            this.regCoeff.setValue(Regression.calcLinReg(this.elevationArray, this.doubles));
        }
    }

    public void cleanup() {
        this.store.close();
    }

    public static int countLeapDays(Attribute.Calendar start, Attribute.Calendar end) {
        int count = 0;
        int startYear = start.get(1);
        int endYear = end.get(1);
        for (int year = startYear; year <= endYear; ++year) {
            GregorianCalendar leapDay;
            if (!Year.of(year).isLeap() || (leapDay = new GregorianCalendar(year, 1, 29)).before(start) || leapDay.after(end)) continue;
            ++count;
        }
        return count;
    }

    public class CustomCalendar360 {
        private static final int DAYS_PER_MONTH = 30;
        private static final int MONTHS_PER_YEAR = 12;
        private static final int DAYS_PER_YEAR = 360;
        private static final int BASE_YEAR = 1970;
        private final LocalDate CUSTOM_EPOCH = LocalDate.of(1970, 1, 1);
        private int year;
        private int month;
        private int day;

        public CustomCalendar360(int year, int month, int day) {
            if (month < 1 || month > 12 || day < 1 || day > 30) {
                throw new IllegalArgumentException("Invalid date in custom calendar");
            }
            this.year = year;
            this.month = month;
            this.day = day;
        }

        public CustomCalendar360 fromGregorian(LocalDate date) {
            long daysSinceEpoch = date.toEpochDay() - this.CUSTOM_EPOCH.toEpochDay();
            int customYear = (int)(daysSinceEpoch / 360L) + 1970;
            int remainingDays = (int)(daysSinceEpoch % 360L);
            if (remainingDays < 0) {
                --customYear;
                remainingDays += 360;
            }
            int customMonth = remainingDays / 30 + 1;
            int customDay = remainingDays % 30 + 1;
            return new CustomCalendar360(customYear, customMonth, customDay);
        }

        public LocalDate toGregorian() {
            long totalDays = (long)(this.year - 1970) * 360L + (long)((this.month - 1) * 30) + (long)(this.day - 1);
            return this.CUSTOM_EPOCH.plusDays(totalDays);
        }

        public void addDays(int days) {
            long totalDays = (long)(this.year - 1970) * 360L + (long)((this.month - 1) * 30) + (long)(this.day - 1) + (long)days;
            this.year = (int)(totalDays / 360L) + 1970;
            int remainingDays = (int)(totalDays % 360L);
            if (remainingDays < 0) {
                --this.year;
                remainingDays += 360;
            }
            this.month = remainingDays / 30 + 1;
            this.day = remainingDays % 30 + 1;
        }

        public long getTimeInMillis() {
            LocalDate gregorianDate = this.toGregorian();
            return gregorianDate.atStartOfDay().toInstant(ZoneOffset.UTC).toEpochMilli();
        }

        public long calculateOffset(CustomCalendar360 other) {
            long totalDaysThis = (long)(this.year - 1970) * 360L + (long)((this.month - 1) * 30) + (long)(this.day - 1);
            long totalDaysOther = (long)(other.year - 1970) * 360L + (long)((other.month - 1) * 30) + (long)(other.day - 1);
            return totalDaysThis - totalDaysOther;
        }

        public String toCustomDateString() {
            return String.format("%04d-%02d-%02d", this.year, this.month, this.day);
        }
    }
}

