/*
 * Decompiled with CFR 0.152.
 */
package ucar.nc2.geotiff;

import java.io.Closeable;
import java.io.IOException;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.RandomAccessFile;
import java.io.StringWriter;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Formatter;
import java.util.List;
import ucar.nc2.geotiff.FieldType;
import ucar.nc2.geotiff.GeoKey;
import ucar.nc2.geotiff.IFDEntry;
import ucar.nc2.geotiff.Tag;
import ucar.nc2.util.CompareNetcdf2;

public class GeoTiff
implements Closeable {
    private static final boolean showBytes = false;
    private static final boolean debugRead = false;
    private static final boolean debugReadGeoKey = false;
    private static final boolean showHeaderBytes = false;
    private String filename;
    private RandomAccessFile file;
    private FileChannel channel;
    private List<IFDEntry> tags = new ArrayList<IFDEntry>();
    private ByteOrder byteOrder = ByteOrder.BIG_ENDIAN;
    private boolean readonly;
    private int headerSize = 8;
    private int firstIFD;
    private int lastIFD;
    private int startOverflowData;
    private int nextOverflowData;
    private List<GeoKey> geokeys = new ArrayList<GeoKey>();

    public GeoTiff(String filename) {
        this.filename = filename;
    }

    @Override
    public void close() throws IOException {
        if (this.channel != null) {
            if (!this.readonly) {
                this.channel.force(true);
                this.channel.truncate(this.nextOverflowData);
            }
            this.channel.close();
        }
        if (this.file != null) {
            this.file.close();
        }
    }

    void addTag(IFDEntry ifd) {
        this.tags.add(ifd);
    }

    List<IFDEntry> getTags() {
        return this.tags;
    }

    void deleteTag(IFDEntry ifd) {
        this.tags.remove(ifd);
    }

    void setTransform(double xStart, double yStart, double xInc, double yInc) {
        this.addTag(new IFDEntry(Tag.ModelTiepointTag, FieldType.DOUBLE).setValue(new double[]{0.0, 0.0, 0.0, xStart, yStart, 0.0}));
        this.addTag(new IFDEntry(Tag.ModelPixelScaleTag, FieldType.DOUBLE).setValue(new double[]{xInc, yInc, 0.0}));
    }

    void addGeoKey(GeoKey geokey) {
        this.geokeys.add(geokey);
    }

    private void writeGeoKeys() {
        if (this.geokeys.isEmpty()) {
            return;
        }
        int extra_chars = 0;
        int extra_ints = 0;
        int extra_doubles = 0;
        for (GeoKey geokey : this.geokeys) {
            if (geokey.isDouble) {
                extra_doubles += geokey.count();
                continue;
            }
            if (geokey.isString) {
                extra_chars += geokey.valueString().length() + 1;
                continue;
            }
            if (geokey.count() <= 1) continue;
            extra_ints += geokey.count();
        }
        int n = (this.geokeys.size() + 1) * 4;
        int[] values = new int[n + extra_ints];
        double[] dvalues = new double[extra_doubles];
        char[] cvalues = new char[extra_chars];
        int icounter = n;
        int dcounter = 0;
        int ccounter = 0;
        values[0] = 1;
        values[1] = 1;
        values[2] = 0;
        values[3] = this.geokeys.size();
        int count = 4;
        for (GeoKey geokey : this.geokeys) {
            int k;
            values[count++] = geokey.tagCode();
            if (geokey.isDouble) {
                values[count++] = Tag.GeoDoubleParamsTag.getCode();
                values[count++] = geokey.count();
                values[count++] = dcounter;
                for (k = 0; k < geokey.count(); ++k) {
                    dvalues[dcounter++] = geokey.valueD(k);
                }
                continue;
            }
            if (geokey.isString) {
                String s2 = geokey.valueString();
                values[count++] = Tag.GeoAsciiParamsTag.getCode();
                values[count++] = s2.length();
                values[count++] = ccounter;
                for (int k2 = 0; k2 < s2.length(); ++k2) {
                    cvalues[ccounter++] = s2.charAt(k2);
                }
                cvalues[ccounter++] = '\u0000';
                continue;
            }
            if (geokey.count() > 1) {
                values[count++] = Tag.GeoKeyDirectoryTag.getCode();
                values[count++] = geokey.count();
                values[count++] = icounter;
                for (k = 0; k < geokey.count(); ++k) {
                    values[icounter++] = geokey.value(k);
                }
                continue;
            }
            values[count++] = 0;
            values[count++] = 1;
            values[count++] = geokey.value();
        }
        this.addTag(new IFDEntry(Tag.GeoKeyDirectoryTag, FieldType.SHORT).setValue(values));
        if (extra_doubles > 0) {
            this.addTag(new IFDEntry(Tag.GeoDoubleParamsTag, FieldType.DOUBLE).setValue(dvalues));
        }
        if (extra_chars > 0) {
            this.addTag(new IFDEntry(Tag.GeoAsciiParamsTag, FieldType.ASCII).setValue(new String(cvalues)));
        }
    }

    int writeData(byte[] data, int imageNumber) throws IOException {
        if (this.file == null) {
            this.init();
        }
        if (imageNumber == 1) {
            this.channel.position(this.headerSize);
        } else {
            this.channel.position(this.nextOverflowData);
        }
        ByteBuffer buffer = ByteBuffer.wrap(data);
        this.channel.write(buffer);
        this.firstIFD = imageNumber == 1 ? this.headerSize + data.length : data.length + this.nextOverflowData;
        return this.nextOverflowData;
    }

    int writeData(float[] data, int imageNumber) throws IOException {
        if (this.file == null) {
            this.init();
        }
        if (imageNumber == 1) {
            this.channel.position(this.headerSize);
        } else {
            this.channel.position(this.nextOverflowData);
        }
        ByteBuffer direct = ByteBuffer.allocateDirect(4 * data.length);
        FloatBuffer buffer = direct.asFloatBuffer();
        buffer.put(data);
        this.channel.write(direct);
        this.firstIFD = imageNumber == 1 ? this.headerSize + 4 * data.length : 4 * data.length + this.nextOverflowData;
        return this.nextOverflowData;
    }

    void writeMetadata(int imageNumber) throws IOException {
        if (this.file == null) {
            this.init();
        }
        this.writeGeoKeys();
        Collections.sort(this.tags);
        if (imageNumber == 1) {
            this.writeHeader(this.channel);
        } else {
            this.channel.position(this.lastIFD);
            ByteBuffer buffer = ByteBuffer.allocate(4);
            buffer.putInt(this.firstIFD);
            ((Buffer)buffer).flip();
            this.channel.write(buffer);
        }
        this.writeIFD(this.channel, this.firstIFD);
    }

    private int writeHeader(FileChannel channel) throws IOException {
        channel.position(0L);
        ByteBuffer buffer = ByteBuffer.allocate(8);
        buffer.put((byte)77);
        buffer.put((byte)77);
        buffer.putShort((short)42);
        buffer.putInt(this.firstIFD);
        ((Buffer)buffer).flip();
        channel.write(buffer);
        return this.firstIFD;
    }

    public void initTags() {
        this.tags = new ArrayList<IFDEntry>();
        this.geokeys = new ArrayList<GeoKey>();
    }

    private void init() throws IOException {
        this.file = new RandomAccessFile(this.filename, "rw");
        this.channel = this.file.getChannel();
        this.readonly = false;
    }

    private void writeIFD(FileChannel channel, int start) throws IOException {
        channel.position(start);
        ByteBuffer buffer = ByteBuffer.allocate(2);
        int n = this.tags.size();
        buffer.putShort((short)n);
        ((Buffer)buffer).flip();
        channel.write(buffer);
        this.nextOverflowData = this.startOverflowData = (start += 2) + 12 * this.tags.size() + 4;
        for (IFDEntry elem : this.tags) {
            this.writeIFDEntry(channel, elem, start);
            start += 12;
        }
        channel.position(this.startOverflowData - 4);
        this.lastIFD = this.startOverflowData - 4;
        buffer = ByteBuffer.allocate(4);
        buffer.putInt(0);
        ((Buffer)buffer).flip();
        channel.write(buffer);
    }

    private void writeIFDEntry(FileChannel channel, IFDEntry ifd, int start) throws IOException {
        channel.position(start);
        ByteBuffer buffer = ByteBuffer.allocate(12);
        buffer.putShort((short)ifd.tag.getCode());
        buffer.putShort((short)ifd.type.code);
        buffer.putInt(ifd.count);
        int size = ifd.count * ifd.type.size;
        if (size <= 4) {
            int done = this.writeValues(buffer, ifd);
            for (int k = 0; k < 4 - done; ++k) {
                buffer.put((byte)0);
            }
            ((Buffer)buffer).flip();
            channel.write(buffer);
        } else {
            buffer.putInt(this.nextOverflowData);
            ((Buffer)buffer).flip();
            channel.write(buffer);
            channel.position(this.nextOverflowData);
            ByteBuffer vbuffer = ByteBuffer.allocate(size);
            this.writeValues(vbuffer, ifd);
            ((Buffer)vbuffer).flip();
            channel.write(vbuffer);
            this.nextOverflowData += size;
        }
    }

    private int writeValues(ByteBuffer buffer, IFDEntry ifd) {
        int done = 0;
        if (ifd.type == FieldType.ASCII) {
            return this.writeSValue(buffer, ifd);
        }
        if (ifd.type == FieldType.RATIONAL) {
            for (int i = 0; i < ifd.count * 2; ++i) {
                done += this.writeIntValue(buffer, ifd, ifd.value[i]);
            }
        } else if (ifd.type == FieldType.FLOAT) {
            for (int i = 0; i < ifd.count; ++i) {
                buffer.putFloat((float)ifd.valueD[i]);
            }
            done += ifd.count * 4;
        } else if (ifd.type == FieldType.DOUBLE) {
            for (int i = 0; i < ifd.count; ++i) {
                buffer.putDouble(ifd.valueD[i]);
            }
            done += ifd.count * 8;
        } else {
            for (int i = 0; i < ifd.count; ++i) {
                done += this.writeIntValue(buffer, ifd, ifd.value[i]);
            }
        }
        return done;
    }

    private int writeIntValue(ByteBuffer buffer, IFDEntry ifd, int v) {
        switch (ifd.type.code) {
            case 1: {
                buffer.put((byte)v);
                return 1;
            }
            case 3: {
                buffer.putShort((short)v);
                return 2;
            }
            case 4: 
            case 5: {
                buffer.putInt(v);
                return 4;
            }
        }
        return 0;
    }

    private int writeSValue(ByteBuffer buffer, IFDEntry ifd) {
        buffer.put(ifd.valueS.getBytes(StandardCharsets.UTF_8));
        int size = ifd.valueS.length();
        if ((size & 1) != 0) {
            ++size;
        }
        return size;
    }

    public void read() throws IOException {
        this.file = new RandomAccessFile(this.filename, "r");
        this.channel = this.file.getChannel();
        this.readonly = true;
        int nextOffset = this.readHeader(this.channel);
        while (nextOffset > 0) {
            nextOffset = this.readIFD(this.channel, nextOffset);
            this.parseGeoInfo();
        }
    }

    IFDEntry findTag(Tag tag) {
        if (tag == null) {
            return null;
        }
        for (IFDEntry ifd : this.tags) {
            if (ifd.tag != tag) continue;
            return ifd;
        }
        return null;
    }

    private int readHeader(FileChannel channel) throws IOException {
        channel.position(0L);
        ByteBuffer buffer = ByteBuffer.allocate(8);
        int n = channel.read(buffer);
        assert (n == 8);
        ((Buffer)buffer).flip();
        byte b = buffer.get();
        if (b == 73) {
            this.byteOrder = ByteOrder.LITTLE_ENDIAN;
        }
        buffer.order(this.byteOrder);
        buffer.position(4);
        int firstIFD = buffer.getInt();
        return firstIFD;
    }

    private int readIFD(FileChannel channel, int start) throws IOException {
        channel.position(start);
        ByteBuffer buffer = ByteBuffer.allocate(2);
        buffer.order(this.byteOrder);
        int n = channel.read(buffer);
        assert (n == 2);
        ((Buffer)buffer).flip();
        int nentries = buffer.getShort();
        start += 2;
        for (int i = 0; i < nentries; ++i) {
            IFDEntry ifd = this.readIFDEntry(channel, start);
            this.tags.add(ifd);
            start += 12;
        }
        channel.position(start);
        buffer = ByteBuffer.allocate(4);
        buffer.order(this.byteOrder);
        assert (4 == channel.read(buffer));
        ((Buffer)buffer).flip();
        int nextIFD = buffer.getInt();
        return nextIFD;
    }

    private IFDEntry readIFDEntry(FileChannel channel, int start) throws IOException {
        channel.position(start);
        ByteBuffer buffer = ByteBuffer.allocate(12);
        buffer.order(this.byteOrder);
        int n = channel.read(buffer);
        assert (n == 12);
        ((Buffer)buffer).flip();
        buffer.position(0);
        int code = this.readUShortValue(buffer);
        Tag tag = Tag.get(code);
        if (tag == null) {
            tag = new Tag(code);
        }
        FieldType type = FieldType.get(this.readUShortValue(buffer));
        int count = buffer.getInt();
        IFDEntry ifd = new IFDEntry(tag, type, count);
        if (ifd.count * ifd.type.size <= 4) {
            this.readValues(buffer, ifd);
        } else {
            int offset = buffer.getInt();
            channel.position(offset);
            ByteBuffer vbuffer = ByteBuffer.allocate(ifd.count * ifd.type.size);
            vbuffer.order(this.byteOrder);
            assert (ifd.count * ifd.type.size == channel.read(vbuffer));
            ((Buffer)vbuffer).flip();
            this.readValues(vbuffer, ifd);
        }
        return ifd;
    }

    private void readValues(ByteBuffer buffer, IFDEntry ifd) {
        if (ifd.type == FieldType.ASCII) {
            ifd.valueS = this.readSValue(buffer, ifd);
        } else if (ifd.type == FieldType.RATIONAL) {
            ifd.value = new int[ifd.count * 2];
            for (int i = 0; i < ifd.count * 2; ++i) {
                ifd.value[i] = this.readIntValue(buffer, ifd);
            }
        } else if (ifd.type == FieldType.FLOAT) {
            ifd.valueD = new double[ifd.count];
            for (int i = 0; i < ifd.count; ++i) {
                ifd.valueD[i] = buffer.getFloat();
            }
        } else if (ifd.type == FieldType.DOUBLE) {
            ifd.valueD = new double[ifd.count];
            for (int i = 0; i < ifd.count; ++i) {
                ifd.valueD[i] = buffer.getDouble();
            }
        } else {
            ifd.value = new int[ifd.count];
            for (int i = 0; i < ifd.count; ++i) {
                ifd.value[i] = this.readIntValue(buffer, ifd);
            }
        }
    }

    private int readIntValue(ByteBuffer buffer, IFDEntry ifd) {
        switch (ifd.type.code) {
            case 1: 
            case 2: {
                return buffer.get();
            }
            case 3: {
                return this.readUShortValue(buffer);
            }
            case 4: 
            case 5: {
                return buffer.getInt();
            }
        }
        return 0;
    }

    private int readUShortValue(ByteBuffer buffer) {
        return buffer.getShort() & 0xFFFF;
    }

    private String readSValue(ByteBuffer buffer, IFDEntry ifd) {
        byte[] dst = new byte[ifd.count];
        buffer.get(dst);
        return new String(dst, StandardCharsets.UTF_8);
    }

    private void printBytes(PrintStream ps, String head, ByteBuffer buffer, int n) {
        ps.print(head + " == ");
        for (int i = 0; i < n; ++i) {
            int b = buffer.get();
            int ub = b < 0 ? b + 256 : b;
            ps.print(ub + "(");
            ps.write(b);
            ps.print(") ");
        }
        ps.println();
    }

    private void parseGeoInfo() {
        IFDEntry keyDir = this.findTag(Tag.GeoKeyDirectoryTag);
        if (null == keyDir) {
            return;
        }
        int nkeys = keyDir.value[3];
        int pos = 4;
        for (int i = 0; i < nkeys; ++i) {
            int id = keyDir.value[pos++];
            int location = keyDir.value[pos++];
            int vcount = keyDir.value[pos++];
            int offset = keyDir.value[pos++];
            GeoKey.Tag tag = GeoKey.Tag.getOrMake(id);
            GeoKey key = null;
            if (location == 0) {
                key = new GeoKey(id, offset);
            } else {
                Object value;
                IFDEntry data = this.findTag(Tag.get(location));
                if (data == null) {
                    System.out.println("********ERROR parseGeoInfo: cant find Tag code = " + location);
                } else if (data.tag == Tag.GeoDoubleParamsTag) {
                    double[] dvalue = new double[vcount];
                    System.arraycopy(data.valueD, offset, dvalue, 0, vcount);
                    key = new GeoKey(tag, dvalue);
                } else if (data.tag == Tag.GeoKeyDirectoryTag) {
                    value = new int[vcount];
                    System.arraycopy(data.value, offset, value, 0, vcount);
                    key = new GeoKey(tag, (int[])value);
                } else if (data.tag == Tag.GeoAsciiParamsTag) {
                    value = data.valueS.substring(offset, offset + vcount);
                    key = new GeoKey(tag, (String)value);
                }
            }
            if (key == null) continue;
            keyDir.addGeoKey(key);
        }
    }

    public void showInfo(PrintWriter out) {
        out.println("Geotiff file= " + this.filename);
        for (int i = 0; i < this.tags.size(); ++i) {
            IFDEntry ifd = this.tags.get(i);
            out.println(i + " IFDEntry == " + ifd);
        }
    }

    public String showInfo() {
        StringWriter sw = new StringWriter(5000);
        this.showInfo(new PrintWriter(sw));
        return sw.toString();
    }

    @Deprecated
    public void compare(GeoTiff other, Formatter f) {
        CompareNetcdf2.compareLists(this.tags, other.getTags(), f);
    }

    ByteBuffer testReadData(int offset, int size) throws IOException {
        this.channel.position(offset);
        ByteBuffer buffer = ByteBuffer.allocate(size);
        buffer.order(this.byteOrder);
        assert (size == this.channel.read(buffer));
        ((Buffer)buffer).flip();
        return buffer;
    }
}

