/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.worldwind.formats.vpf;

import gov.nasa.worldwind.exception.WWRuntimeException;
import gov.nasa.worldwind.formats.vpf.VPFBufferedRecordData;
import gov.nasa.worldwind.formats.vpf.VPFDataBuffer;
import gov.nasa.worldwind.formats.vpf.VPFDataType;
import gov.nasa.worldwind.formats.vpf.VPFUtils;
import gov.nasa.worldwind.util.Logging;
import gov.nasa.worldwind.util.WWIO;
import gov.nasa.worldwind.util.WWUtil;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.logging.Level;

public class VPFTableReader {
    public VPFBufferedRecordData read(File file) {
        if (file == null) {
            String string = Logging.getMessage("nullValue.FileIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        try {
            ByteBuffer byteBuffer = this.readFileToBuffer(file);
            return this.doRead(file, byteBuffer);
        }
        catch (Exception exception) {
            String string = Logging.getMessage("VPF.ExceptionAttemptingToReadTable", file.getPath());
            Logging.logger().log(Level.SEVERE, string, exception);
            throw new WWRuntimeException(string, exception);
        }
    }

    protected ByteBuffer readFileToBuffer(File file) throws IOException {
        ByteBuffer byteBuffer = WWIO.readFileToBuffer(file, true);
        byteBuffer.order(ByteOrder.LITTLE_ENDIAN);
        return byteBuffer;
    }

    protected VPFBufferedRecordData doRead(File file, ByteBuffer byteBuffer) {
        Header header = this.readHeader(byteBuffer);
        byteBuffer.order(header.byteOrder);
        RecordIndex recordIndex = null;
        File file2 = new File(file.getParent(), VPFTableReader.getRecordIndexFilename(file.getName()));
        if (file2.exists()) {
            recordIndex = this.readRecordIndex(file2);
        }
        if (recordIndex == null) {
            recordIndex = this.computeRecordIndex(byteBuffer, header);
        }
        if (recordIndex == null) {
            String string = Logging.getMessage("VPF.VariableLengthIndexFileMissing");
            Logging.logger().severe(string);
            throw new WWRuntimeException(string);
        }
        return this.readRecordData(byteBuffer, header.columns, recordIndex);
    }

    protected Header readHeader(ByteBuffer byteBuffer) {
        int n = byteBuffer.position();
        int n2 = byteBuffer.getInt();
        Header header = new Header();
        header.length = n2;
        header.byteOrder = ByteOrder.LITTLE_ENDIAN;
        if (n2 == 0) {
            return header;
        }
        String string = VPFUtils.readDelimitedText(byteBuffer, ';');
        if (string != null && string.equalsIgnoreCase("M")) {
            header.byteOrder = ByteOrder.BIG_ENDIAN;
        }
        if ((string = VPFUtils.readDelimitedText(byteBuffer, ';')) != null) {
            header.description = string.trim();
        }
        if ((string = VPFUtils.readDelimitedText(byteBuffer, ';')) != null && string.charAt(0) != '-') {
            header.narrativeTableName = string.trim();
        }
        ArrayList<Column> arrayList = new ArrayList<Column>();
        while (byteBuffer.position() < n + n2) {
            Column column = this.readColumnDescription(byteBuffer);
            arrayList.add(column);
        }
        header.columns = new Column[arrayList.size()];
        arrayList.toArray(header.columns);
        VPFUtils.readDelimitedText(byteBuffer, ';');
        return header;
    }

    protected Column readColumnDescription(ByteBuffer byteBuffer) {
        int n;
        String string = VPFUtils.readDelimitedText(byteBuffer, '=');
        if (string == null) {
            String string2 = Logging.getMessage("VPF.MissingColumnName");
            Logging.logger().severe(string2);
            throw new WWRuntimeException(string2);
        }
        Column column = new Column(string);
        string = VPFUtils.readDelimitedText(byteBuffer, ',');
        if (string != null) {
            column.dataType = string;
        }
        if ((string = VPFUtils.readDelimitedText(byteBuffer, ',')) != null) {
            column.numElements = VPFTableReader.parseNumElements(string);
        }
        if ((string = VPFUtils.readDelimitedText(byteBuffer, ',')) != null) {
            column.keyType = string;
        }
        if ((string = VPFUtils.readDelimitedText(byteBuffer, ',')) != null) {
            column.description = string;
        }
        if ((string = VPFUtils.readDelimitedText(byteBuffer, ',')) != null) {
            column.valueDescriptionTableName = string;
        }
        if ((string = VPFUtils.readDelimitedText(byteBuffer, ',')) != null) {
            column.thematicIndexName = string;
        }
        if ((string = VPFUtils.readDelimitedText(byteBuffer, ':')) != null && (n = string.indexOf(",")) >= 0) {
            column.narrativeTableName = string = string.substring(0, n);
        }
        return column;
    }

    protected static int parseNumElements(String string) {
        if (string == null || string.equals("*")) {
            return -1;
        }
        Integer n = WWUtil.convertStringToInteger(string);
        return n != null ? n : -1;
    }

    protected VPFBufferedRecordData readRecordData(ByteBuffer byteBuffer, Column[] columnArray, RecordIndex recordIndex) {
        int n;
        int n2 = recordIndex.numEntries;
        int n3 = columnArray.length;
        RecordDataReader[] recordDataReaderArray = new RecordDataReader[n3];
        for (n = 0; n < n3; ++n) {
            VPFDataType vPFDataType = VPFDataType.fromTypeName(columnArray[n].dataType);
            VPFDataBuffer vPFDataBuffer = vPFDataType.createDataBuffer(n2, columnArray[n].numElements);
            recordDataReaderArray[n] = columnArray[n].isVariableLengthField() ? new VariableLengthDataReader(vPFDataBuffer) : new FixedLengthDataReader(vPFDataBuffer, columnArray[n].numElements);
        }
        for (n = 0; n < n2; ++n) {
            byteBuffer.position(recordIndex.entries[n].offset);
            for (int i = 0; i < n3; ++i) {
                recordDataReaderArray[i].read(byteBuffer);
            }
        }
        VPFBufferedRecordData vPFBufferedRecordData = new VPFBufferedRecordData();
        vPFBufferedRecordData.setNumRecords(n2);
        for (int i = 0; i < n3; ++i) {
            vPFBufferedRecordData.setRecordData(columnArray[i].name, recordDataReaderArray[i].getDataBuffer());
            if (columnArray[i].name.equals("id") || !columnArray[i].name.equals("P") && !columnArray[i].name.equals("U")) continue;
            vPFBufferedRecordData.buildRecordIndex(columnArray[i].name);
        }
        return vPFBufferedRecordData;
    }

    protected static String getRecordIndexFilename(String string) {
        boolean bl = string.equalsIgnoreCase("fcs");
        StringBuilder stringBuilder = new StringBuilder();
        int n = string.length();
        stringBuilder.append(string, 0, n > 0 ? n - 1 : n);
        stringBuilder.append(bl ? "z" : "x");
        return stringBuilder.toString();
    }

    protected RecordIndex readRecordIndex(File file) {
        try {
            ByteBuffer byteBuffer = this.readFileToBuffer(file);
            byteBuffer.order(ByteOrder.LITTLE_ENDIAN);
            RecordIndex recordIndex = new RecordIndex();
            recordIndex.numEntries = byteBuffer.getInt();
            recordIndex.headerLength = byteBuffer.getInt();
            recordIndex.entries = new RecordIndex.Entry[recordIndex.numEntries];
            for (int i = 0; i < recordIndex.numEntries; ++i) {
                int n = byteBuffer.getInt();
                int n2 = byteBuffer.getInt();
                recordIndex.entries[i] = new RecordIndex.Entry(n, n2);
            }
            return recordIndex;
        }
        catch (Exception exception) {
            String string = Logging.getMessage("VPF.ExceptionAttemptingToReadRecordIndex", file.getPath());
            Logging.logger().log(Level.SEVERE, string, exception);
            throw new WWRuntimeException(string, exception);
        }
    }

    protected RecordIndex computeRecordIndex(ByteBuffer byteBuffer, Header header) {
        int n = 0;
        for (Column object2 : header.columns) {
            if (object2.isVariableLengthField()) {
                return null;
            }
            n += object2.getFieldLength();
        }
        int n2 = 4 + header.length;
        int n3 = byteBuffer.limit() - n2;
        int n4 = n3 / n;
        RecordIndex recordIndex = new RecordIndex();
        recordIndex.headerLength = header.length;
        recordIndex.numEntries = n4;
        recordIndex.entries = new RecordIndex.Entry[n4];
        int n5 = n2;
        for (int i = 0; i < n4; ++i) {
            recordIndex.entries[i] = new RecordIndex.Entry(n5, n);
            n5 += recordIndex.entries[i].length;
        }
        return recordIndex;
    }

    public static class RecordIndex {
        public int numEntries;
        public int headerLength;
        public Entry[] entries;

        public static class Entry {
            public int offset;
            public int length;

            public Entry(int n, int n2) {
                this.offset = n;
                this.length = n2;
            }
        }
    }

    protected static class VariableLengthDataReader
    extends AbstractDataReader {
        public VariableLengthDataReader(VPFDataBuffer vPFDataBuffer) {
            super(vPFDataBuffer);
        }

        @Override
        public void read(ByteBuffer byteBuffer) {
            this.dataBuffer.read(byteBuffer);
        }
    }

    protected static class FixedLengthDataReader
    extends AbstractDataReader {
        protected int numElements;

        public FixedLengthDataReader(VPFDataBuffer vPFDataBuffer, int n) {
            super(vPFDataBuffer);
            this.numElements = n;
        }

        @Override
        public void read(ByteBuffer byteBuffer) {
            this.dataBuffer.read(byteBuffer, this.numElements);
        }
    }

    protected static abstract class AbstractDataReader
    implements RecordDataReader {
        protected VPFDataBuffer dataBuffer;

        public AbstractDataReader(VPFDataBuffer vPFDataBuffer) {
            this.dataBuffer = vPFDataBuffer;
        }

        @Override
        public VPFDataBuffer getDataBuffer() {
            return this.dataBuffer;
        }
    }

    protected static interface RecordDataReader {
        public VPFDataBuffer getDataBuffer();

        public void read(ByteBuffer var1);
    }

    public class Column {
        public final String name;
        public String dataType;
        public int numElements;
        public String keyType;
        public String description;
        public String valueDescriptionTableName;
        public String thematicIndexName;
        public String narrativeTableName;
        public VPFDataBuffer dataBuffer;

        public Column(String string) {
            this.name = string;
        }

        public int getFieldLength() {
            if (this.isVariableLengthField()) {
                return -1;
            }
            VPFDataType vPFDataType = VPFDataType.fromTypeName(this.dataType);
            return this.numElements * vPFDataType.getFieldLength();
        }

        public boolean isVariableLengthField() {
            VPFDataType vPFDataType = VPFDataType.fromTypeName(this.dataType);
            return this.numElements < 0 || vPFDataType.isVariableLength();
        }

        public boolean isPrimaryKey() {
            return this.keyType.equals("P");
        }

        public boolean isUniqueKey() {
            return this.keyType.equals("U");
        }

        public boolean isNonUniqueKey() {
            return this.keyType.equals("N");
        }
    }

    protected static class Header {
        public int length;
        public ByteOrder byteOrder = ByteOrder.LITTLE_ENDIAN;
        public String description;
        public String narrativeTableName;
        public Column[] columns;
    }
}

