/*
 * Decompiled with CFR 0.152.
 */
package uk.ac.rdg.resc.ncwms.cdm;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import org.geotoolkit.metadata.iso.extent.DefaultGeographicBoundingBox;
import org.joda.time.Chronology;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.opengis.metadata.extent.GeographicBoundingBox;
import org.opengis.referencing.operation.TransformException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.ma2.Index;
import ucar.ma2.InvalidRangeException;
import ucar.ma2.Range;
import ucar.nc2.Attribute;
import ucar.nc2.constants.FeatureType;
import ucar.nc2.dataset.CoordinateAxis1D;
import ucar.nc2.dataset.CoordinateAxis1DTime;
import ucar.nc2.dataset.NetcdfDataset;
import ucar.nc2.dataset.VariableEnhanced;
import ucar.nc2.dt.GridCoordSystem;
import ucar.nc2.dt.GridDataset;
import ucar.nc2.dt.GridDatatype;
import ucar.nc2.dt.TypedDatasetFactory;
import ucar.unidata.geoloc.LatLonPointImpl;
import ucar.unidata.geoloc.LatLonRect;
import uk.ac.rdg.resc.ncwms.cdm.DataChunk;
import uk.ac.rdg.resc.ncwms.cdm.DataReadingStrategy;
import uk.ac.rdg.resc.ncwms.cdm.LayerBuilder;
import uk.ac.rdg.resc.ncwms.cdm.RangesList;
import uk.ac.rdg.resc.ncwms.coords.CrsHelper;
import uk.ac.rdg.resc.ncwms.coords.HorizontalCoordSys;
import uk.ac.rdg.resc.ncwms.coords.HorizontalPosition;
import uk.ac.rdg.resc.ncwms.coords.LonLatPosition;
import uk.ac.rdg.resc.ncwms.coords.PixelMap;
import uk.ac.rdg.resc.ncwms.coords.PointList;
import uk.ac.rdg.resc.ncwms.coords.chrono.ThreeSixtyDayChronology;
import uk.ac.rdg.resc.ncwms.util.TimeUtils;
import uk.ac.rdg.resc.ncwms.wms.ScalarLayer;

public final class CdmUtils {
    private static final Logger logger = LoggerFactory.getLogger(CdmUtils.class);
    static final Range ZERO_RANGE;

    private CdmUtils() {
        throw new AssertionError();
    }

    public static <L extends ScalarLayer> void findAndUpdateLayers(GridDataset gd, LayerBuilder<L> layerBuilder, Map<String, L> layers) {
        if (gd == null) {
            throw new NullPointerException("GridDataset can't be null");
        }
        if (layerBuilder == null) {
            throw new NullPointerException("LayerBuilder can't be null");
        }
        if (layers == null) {
            throw new NullPointerException("layers can't be null");
        }
        for (GridDataset.Gridset gridset : gd.getGridsets()) {
            GridCoordSystem coordSys = gridset.getGeoCoordSystem();
            List grids = gridset.getGrids();
            ArrayList<GridDatatype> newGrids = new ArrayList<GridDatatype>();
            for (GridDatatype grid : grids) {
                if (layers.containsKey(grid.getName())) {
                    logger.debug("We already have data for {}", (Object)grid.getName());
                    continue;
                }
                logger.debug("{} is a new grid", (Object)grid.getName());
                newGrids.add(grid);
            }
            if (!newGrids.isEmpty()) {
                logger.debug("Creating coordinate system objects");
                HorizontalCoordSys horizCoordSys = HorizontalCoordSys.fromCoordSys(coordSys);
                boolean zPositive = coordSys.isZPositive();
                CoordinateAxis1D zAxis = coordSys.getVerticalAxis();
                List<Double> zValues = CdmUtils.getZValues(zAxis, zPositive);
                GeographicBoundingBox bbox = CdmUtils.getBbox(coordSys);
                for (GridDatatype grid : newGrids) {
                    logger.debug("Creating new Layer object for {}", (Object)grid.getName());
                    L layer = layerBuilder.newLayer(grid.getName());
                    layerBuilder.setTitle((ScalarLayer)layer, CdmUtils.getLayerTitle((VariableEnhanced)grid.getVariable()));
                    layerBuilder.setAbstract((ScalarLayer)layer, grid.getDescription());
                    layerBuilder.setUnits((ScalarLayer)layer, grid.getUnitsString());
                    layerBuilder.setHorizontalCoordSys((ScalarLayer)layer, horizCoordSys);
                    layerBuilder.setGeographicBoundingBox((ScalarLayer)layer, bbox);
                    layerBuilder.setGridDatatype((ScalarLayer)layer, grid);
                    if (zAxis != null) {
                        layerBuilder.setElevationAxis((ScalarLayer)layer, zValues, zPositive, zAxis.getUnitsString());
                    }
                    layers.put(layer.getId(), layer);
                }
            }
            List<DateTime> timesteps = CdmUtils.getTimesteps(coordSys);
            for (GridDatatype grid : grids) {
                ScalarLayer layer = (ScalarLayer)layers.get(grid.getName());
                layerBuilder.setTimeValues(layer, timesteps);
            }
        }
    }

    public static GridDataset getGridDataset(NetcdfDataset nc) throws IOException {
        return (GridDataset)TypedDatasetFactory.open((FeatureType)FeatureType.GRID, (NetcdfDataset)nc, null, null);
    }

    public static DataReadingStrategy getOptimumDataReadingStrategy(NetcdfDataset nc) {
        String fileType = nc.getFileTypeId();
        return "netCDF".equals(fileType) || "HDF4".equals(fileType) ? DataReadingStrategy.SCANLINE : DataReadingStrategy.BOUNDING_BOX;
    }

    private static GeographicBoundingBox getBbox(GridCoordSystem coordSys) {
        LatLonRect latLonRect = coordSys.getLatLonBoundingBox();
        LatLonPointImpl lowerLeft = latLonRect.getLowerLeftPoint();
        LatLonPointImpl upperRight = latLonRect.getUpperRightPoint();
        double minLon = lowerLeft.getLongitude();
        double maxLon = upperRight.getLongitude();
        double minLat = lowerLeft.getLatitude();
        double maxLat = upperRight.getLatitude();
        if (latLonRect.crossDateline() || minLon >= maxLon) {
            minLon = -180.0;
            maxLon = 180.0;
        }
        if (minLat >= maxLat) {
            minLat = -90.0;
            maxLat = 90.0;
        }
        minLon = Double.isNaN(minLon) ? -180.0 : minLon;
        minLat = Double.isNaN(minLat) ? -90.0 : minLat;
        maxLon = Double.isNaN(maxLon) ? 180.0 : maxLon;
        maxLat = Double.isNaN(maxLat) ? 90.0 : maxLat;
        return new DefaultGeographicBoundingBox(minLon, maxLon, minLat, maxLat);
    }

    private static String getLayerTitle(VariableEnhanced var) {
        Attribute stdNameAtt = var.findAttributeIgnoreCase("standard_name");
        if (stdNameAtt == null || stdNameAtt.getStringValue().trim().equals("")) {
            Attribute longNameAtt = var.findAttributeIgnoreCase("long_name");
            if (longNameAtt == null || longNameAtt.getStringValue().trim().equals("")) {
                return var.getName();
            }
            return longNameAtt.getStringValue();
        }
        return stdNameAtt.getStringValue();
    }

    private static List<Double> getZValues(CoordinateAxis1D zAxis, boolean zPositive) {
        ArrayList<Double> zValues = new ArrayList<Double>();
        if (zAxis != null) {
            for (double zVal : zAxis.getCoordValues()) {
                zValues.add(zPositive ? zVal : 0.0 - zVal);
            }
        }
        return zValues;
    }

    private static List<DateTime> getTimesteps(GridCoordSystem coordSys) {
        if (coordSys.hasTimeAxis1D()) {
            String calString;
            CoordinateAxis1DTime timeAxis = coordSys.getTimeAxis1D();
            Attribute cal = timeAxis.findAttribute("calendar");
            String string = calString = cal == null ? null : cal.getStringValue().toLowerCase();
            if (calString == null || calString.equals("gregorian") || calString.equals("standard")) {
                ArrayList<DateTime> timesteps = new ArrayList<DateTime>();
                for (Date date : timeAxis.getTimeDates()) {
                    timesteps.add(new DateTime((Object)date, DateTimeZone.UTC));
                }
                return timesteps;
            }
            if (calString.equals("360_day")) {
                return CdmUtils.getTimesteps360Day(timeAxis);
            }
            throw new IllegalArgumentException("The calendar system " + cal.getStringValue() + " cannot be handled");
        }
        return Collections.emptyList();
    }

    private static List<DateTime> getTimesteps360Day(CoordinateAxis1DTime timeAxis) {
        String timeAxisUnits = timeAxis.getUnitsString();
        int indexOfSince = timeAxisUnits.indexOf(" since ");
        String unitIncrement = timeAxisUnits.substring(0, indexOfSince);
        long unitLength = TimeUtils.getUnitLengthMillis(unitIncrement);
        String baseDateTimeString = timeAxisUnits.substring(indexOfSince + " since ".length());
        DateTime baseDateTime = TimeUtils.parseUdunitsTimeString(baseDateTimeString, (Chronology)ThreeSixtyDayChronology.getInstanceUTC());
        ArrayList<DateTime> timesteps = new ArrayList<DateTime>();
        for (double val : timeAxis.getCoordValues()) {
            timesteps.add(baseDateTime.plus((long)((double)unitLength * val)));
        }
        return timesteps;
    }

    public static List<Float> readPointList(GridDatatype grid, HorizontalCoordSys horizCoordSys, int tIndex, int zIndex, PointList pointList, DataReadingStrategy drStrategy) throws IOException {
        try {
            ArrayList<Float> picData = CdmUtils.nullArrayList(pointList.size());
            long start = System.currentTimeMillis();
            PixelMap pixelMap = new PixelMap(horizCoordSys, pointList);
            if (pixelMap.isEmpty()) {
                return picData;
            }
            long readMetadata = System.currentTimeMillis();
            logger.debug("Created PixelMap in {} milliseconds", (Object)(readMetadata - start));
            drStrategy.populatePixelArray(picData, tIndex, zIndex, pixelMap, grid);
            long builtPic = System.currentTimeMillis();
            logger.debug("Built picture array in {} milliseconds", (Object)(builtPic - readMetadata));
            logger.debug("Whole read() operation took {} milliseconds", (Object)(builtPic - start));
            return picData;
        }
        catch (TransformException te) {
            throw new RuntimeException(te);
        }
    }

    public static List<Float> readTimeseries(GridDatatype grid, HorizontalCoordSys horizCoordSys, List<Integer> tIndices, int zIndex, HorizontalPosition xy, boolean scaleMissingDeferred) throws IOException {
        LonLatPosition lonLat;
        if (xy instanceof LonLatPosition) {
            lonLat = (LonLatPosition)xy;
        } else {
            if (xy.getCoordinateReferenceSystem() == null) {
                throw new IllegalArgumentException("Horizontal position must have a coordinate reference system");
            }
            CrsHelper crsHelper = CrsHelper.fromCrs(xy.getCoordinateReferenceSystem());
            try {
                lonLat = crsHelper.crsToLonLat(xy);
            }
            catch (TransformException te) {
                throw new RuntimeException(te);
            }
        }
        int[] gridCoords = horizCoordSys.lonLatToGrid(lonLat);
        if (gridCoords == null) {
            return Collections.nCopies(tIndices.size(), null);
        }
        int firstTIndex = tIndices.get(0);
        int lastTIndex = tIndices.get(tIndices.size() - 1);
        if (firstTIndex < 0 || lastTIndex < 0) {
            firstTIndex = 0;
            lastTIndex = 0;
        }
        if (zIndex < 0) {
            zIndex = 0;
        }
        RangesList rangesList = new RangesList(grid);
        rangesList.setTRange(firstTIndex, lastTIndex);
        rangesList.setZRange(zIndex, zIndex);
        rangesList.setYRange(gridCoords[1], gridCoords[1]);
        rangesList.setXRange(gridCoords[0], gridCoords[0]);
        DataChunk dataChunk = DataChunk.readDataChunk(grid.getVariable(), rangesList);
        ArrayList<Float> tsData = new ArrayList<Float>();
        Index index = dataChunk.getIndex();
        index.set(new int[index.getRank()]);
        for (int tIndex : tIndices) {
            int tIndexOffset = tIndex - firstTIndex;
            if (tIndexOffset < 0) {
                tIndexOffset = 0;
            }
            index.setDim(rangesList.getTAxisIndex(), tIndexOffset);
            float val = dataChunk.readFloatValue(index);
            tsData.add(Float.isNaN(val) ? null : Float.valueOf(val));
        }
        return tsData;
    }

    public static boolean isScaleMissingDeferred(NetcdfDataset nc) {
        return nc.getEnhanceMode().contains(NetcdfDataset.Enhance.ScaleMissingDefer);
    }

    private static ArrayList<Float> nullArrayList(int n) {
        ArrayList<Float> list = new ArrayList<Float>(n);
        for (int i = 0; i < n; ++i) {
            list.add(null);
        }
        return list;
    }

    static {
        try {
            ZERO_RANGE = new Range(0, 0);
        }
        catch (InvalidRangeException invalidRangeException) {
            // empty catch block
        }
    }
}

